import os

from fastapi import FastAPI, Header, HTTPException
from pydantic import BaseModel

from ..agent_core import Agent  # noqa: TID252

# =============================================================================
# USING YOUR OWN AGENT ENDPOINT
# =============================================================================
#
# This example uses a LangGraph-based agent defined in agent.py. To connect
# your own agent running at an external endpoint instead:
#
# 1. Remove the `from .agent import Agent` import above
# 2. Add `import httpx` to the imports
# 3. Replace the code between --- AGENT CHAT LOGIC STARTS HERE --- and
#    --- AGENT CHAT LOGIC ENDS HERE --- with:
#
#    AGENT_ENDPOINT = "<YOUR_AGENT_ENDPOINT>"  # e.g., "http://localhost:8080/chat"
#
#    payload = {
#        "id": chat_id,
#        "question": last_user_msg.content
#    }
#
#    async with httpx.AsyncClient(timeout=30.0) as client:
#        agent_response = await client.post(
#            AGENT_ENDPOINT,
#            json=payload,
#            headers={"Content-Type": "application/json"}
#        )
#
#    if agent_response.status_code >= 400:
#        raise HTTPException(agent_response.status_code, f"Agent error: {agent_response.text}")
#
#    response = agent_response.json()
#    answer_text = response["answer"]
#
# Note: Adjust the payload structure and response parsing to match your agent's API.
# =============================================================================

MY_API_KEY = os.getenv("MY_API_KEY", "1234567890")
app = FastAPI(title="Chat Completion Wrapper")


# OpenAI request body
class ChatCompletionRequestMessage(BaseModel):
    role: str
    content: str


class ChatCompletionRequest(BaseModel):
    messages: list[ChatCompletionRequestMessage]
    metadata: dict | None = None


class ChatCompletionResponse(BaseModel):
    choices: list[dict]


# Cache for agent graphs by chat_id
_agent_cache: dict[str, Agent] = {}


@app.post("/chat/completions")
async def chat_completions(  # noqa: ANN201
    request: ChatCompletionRequest, authorization: str | None = Header(None)
):
    # Authorization Check
    if not authorization or not authorization.startswith("Bearer "):
        raise HTTPException(status_code=401, detail="Missing Bearer token")

    token = authorization.split(" ")[1]
    if token != MY_API_KEY:
        raise HTTPException(status_code=403, detail="Invalid API token")

    # Get the chat id from the request
    chat_id = request.messages[0].content.split("chat_id:")[1].split(" ")[0]
    if not chat_id:
        # If chat_id is not provided, we cannot make the chat request
        raise HTTPException(status_code=400, detail="Chat ID is required")

    # Find the last user message from the messages which contins the simulator query
    try:
        last_user_msg = next(m for m in reversed(request.messages) if m.role == "user")
    except StopIteration:
        raise HTTPException(status_code=400, detail="No user message found in request.")  # noqa: B904

    # --- AGENT CHAT LOGIC STARTS HERE ---
    # Get or create agent instance for this chat_id
    if chat_id not in _agent_cache:
        agent = Agent(context_id=chat_id)
        _agent_cache[chat_id] = agent
    else:
        agent = _agent_cache[chat_id]

    answer_text = await agent.invoke(last_user_msg.content)
    # --- AGENT CHAT LOGIC ENDS HERE ---

    # Format response to chat completion format
    return ChatCompletionResponse(
        choices=[
            {
                "message": {"role": "assistant", "content": answer_text},
            }
        ],
    )


if __name__ == "__main__":
    import uvicorn

    uvicorn.run(app, host="0.0.0.0", port=8080)
